/* 
 * This is JavaSQL, a tool to check the validity of connections and queries
 * in JDBC.
 * Copyright (C) 2002 Alexander Lindhorst
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact:
 * Alexander Lindhorst
 * Elsa-Brandstrm-Weg 3
 * 33102 Paderborn
 * Germany
 * al@alexander-lindhorst.de
 */


package lindhorst.apps.jdbc.swing;

import javax.swing.*;
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import lindhorst.apps.jdbc.swing.helpers.*;
import lindhorst.apps.jdbc.swing.data.*;
import lindhorst.apps.jdbc.swing.modules.*;
import lindhorst.apps.jdbc.swing.modules.datamodels.*;
import lindhorst.apps.jdbc.swing.administration.*;
import lindhorst.apps.jdbc.swing.event.*;
import lindhorst.apps.jdbc.swing.info.*;

public class AppPane extends JDesktopPane implements ModuleRequestListener {
    public static final Integer APPLICATION_LAYER=new Integer(500);
    public static final Integer DIALOG_LAYER=new Integer(1000);
    public static final Integer ADMINISTRATION_LAYER=new Integer(250);
    
    private Component focusedComponent=null;
    private SessionDataModel model=null;
    
    public AppPane() {
        putClientProperty("JDesktopPane.dragMode", "outline");
    }
    
    public void createSession() {
        createSession(null);
    }
    
    public void createSession(Properties props) {
        closeAll();
        LogInDialog dialog=null;
        try {
            if(model!=null) {
                model.close();
                model.removeModuleRequestListener(this);
            }
            
            AppPane.LoginStatusWatch watch=new AppPane.LoginStatusWatch();
            
            dialog=new LogInDialog(Helpers.getTopLevelContainer());
            if(props!=null)dialog.setProperties(props);
            dialog.setVisible(true);
            if(!dialog.isProperlyClosed()) {
                throw new NullPointerException(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("action_aborted.no_properties"));
            }
            
            //make sure! (Java 2)
            dialog.setVisible(false);
            props=dialog.getProperties();
            
            model=new SessionDataModel(props,watch);
            model.addModuleRequestListener(this);
            ModuleAdministration admin=model.getAdministrationPanel();
            Dimension adminPSize=admin.getPreferredSize();
            Dimension size=getSize();
            admin.setBounds((size.width-adminPSize.width)/2,(size.height-adminPSize.height)/2,adminPSize.width,adminPSize.height);
            add(admin,ADMINISTRATION_LAYER);
            admin.setVisible(true);
            admin.setSelected(true);
            Helpers.toFront();
        }
        catch(NullPointerException ne) {
            //Nur wenn "Abbrechen"
            ne.printStackTrace(System.err);
            if(dialog.isProperlyClosed()) {
                //doch nicht Abbrechen
                Helpers.showError(ne);
            }
        }
        catch(Exception e) {
            e.printStackTrace(System.err);
            Helpers.showError(e);
        }
    }
    
    public void closeSession() {
        try {
            closeAll();
        }
        catch(Exception e) {
            Helpers.showError(e);
        }
        finally {
            removeAll();
            focusedComponent=null;
        }
    }
    
    public void closeAll() {
        JInternalFrame[] frames=getAllFrames();
        for(int i=0;i<frames.length;i++)
            frames[i].dispose();
        try {
            if(model!=null) {
                model.close();
                model=null;
            }
        }
        catch(Exception e) {
            Helpers.showError(e);
        }
        repaint();
        System.gc();
    }
    
    public SessionDataModel getModel() {
        return model;
    }
    
    public boolean isOpaque() {
        return true;
    }
    
    public void moduleRequested(ModuleRequestEvent event) {
        String type=event.getModuleType();
        String name=event.getModuleName();
        if(event.getModuleType().equals(Module.QUERY)) {
            if(name.equals("<Neu>")) {
                try {
                    QueryMaker queries=new QueryMaker(model.getReadableConnection(),model.getWritableConnection());
                    Dimension size=getSize();
                    queries.setBounds(0,0,size.width,size.height);
                    add(queries,APPLICATION_LAYER);
                    queries.setSelected(true);
                }
                catch(Exception e) {
                    Helpers.showError(e);
                }
                
                return;
            }
            
            try {
                QueriesDataModel qModel=new QueriesDataModel(model.getReadableConnection(),model.getWritableConnection());
                qModel.executeSQL(model.getProperties().getProperty("jdbcsession."+Module.QUERY+"."+name+".command"));
                Queries queries=new Queries(qModel,name);
                Dimension size=getSize();
                queries.setBounds(0,0,size.width,size.height);
                add(queries,APPLICATION_LAYER);
                queries.setSelected(true);
            }
            catch(Exception e) {
                Helpers.showError(e);
            }
        }
        else {
            if(name.equals("<Neu>")) {
                try {
                    name=JOptionPane.showInputDialog(this,java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("enter_table_name"),java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("table_name"),JOptionPane.QUESTION_MESSAGE);
                    if(name==null || name.equals("")) return;
                }
                catch(Exception e) {
                    Helpers.showError(e);
                }
            }
            
            try {
                TableDataModel tModel=new TableDataModel(model.getReadableConnection(),model.getWritableConnection(),name);
                Table table=new Table(tModel);
                Dimension size=getSize();
                table.setBounds(0,0,size.width,size.height);
                add(table,APPLICATION_LAYER);
                table.setSelected(true);
            }
            catch(Exception e) {
                Helpers.showError(e);
            }
        }
    }
    
    //relative to Screen
    private Dimension prefSize=null;
    public Dimension getPreferredSize() {
        if(prefSize==null) {
            Dimension screen=Toolkit.getDefaultToolkit().getScreenSize();
            int w=screen.width*7/8;
            int h=screen.height*3/4;
            prefSize=new Dimension(w,h);
        }
        return prefSize;
    }
    
    private AppPane getSelfReference() {
        return this;
    }
    
    private class FocusWatch extends FocusAdapter {
        public void focusGained(FocusEvent event) {
            focusedComponent=(Component)event.getSource();
        }
    }
    
    private class LoginStatusWatch implements LoginListener,Runnable{
        private LoggingStatusDialog dialog=null;
        
        LoginStatusWatch() {
            dialog=new LoggingStatusDialog();
        }
        
        public void loginStatusChanged(LoginEvent event) {
            dialog.setDialogType(event.getType());
            if(!SwingUtilities.isEventDispatchThread()) {
                try{
                    SwingUtilities.invokeAndWait(this);
                }
                catch(Exception ex) {
                    ex.printStackTrace(System.err);
                }
                Thread.yield();
            }
            else {
                switch(dialog.getDialogType()) {
                    case LoginEvent.LOGIN_EVENT:
                    case LoginEvent.LOGOUT_EVENT:
                        setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
                        dialog.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
                        dialog.reactToEvent();
                        break;
                    default: //*_COMPLETED
                        setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
                        dialog.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
                        dialog.reactToEvent();
                        synchronized(dialog) {
                            dialog.dispose();
                            dialog=new LoggingStatusDialog(); //make window disappear in X11
                            dialog.pack();
                        }
                }
                Exception e=model.getRaisedException();
                if(model !=null &&e!=null){
                    Helpers.showError(e);
                    closeAll();
                }
            }
        }
        
        /** When an object implementing interface <code>Runnable</code> is used
         * to create a thread, starting the thread causes the object's
         * <code>run</code> method to be called in that separately executing
         * thread.
         * <p>
         * The general contract of the method <code>run</code> is that it may
         * take any action whatsoever.
         *
         * @see     java.lang.Thread#run()
         */
        public void run() {
            switch(dialog.getDialogType()) {
                case LoginEvent.LOGIN_EVENT:
                case LoginEvent.LOGOUT_EVENT:
                    setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
                    dialog.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
                    dialog.reactToEvent();
                     break;
                default:
                    setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
                    dialog.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
                    dialog.reactToEvent();
                    synchronized(dialog) {
                        dialog.dispose();
                        dialog=new LoggingStatusDialog(); //make window disappear in X11
                        dialog.pack();
                    }
            }
            if(model!=null && model.getRaisedException()!=null){
                Helpers.showError(model.getRaisedException());
                closeAll();
            }
        }
        
    }
}
