/* 
 * This is JavaSQL, a tool to check the validity of connections and queries
 * in JDBC.
 * Copyright (C) 2002 Alexander Lindhorst
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact:
 * Alexander Lindhorst
 * Elsa-Brandstrm-Weg 3
 * 33102 Paderborn
 * Germany
 * al@alexander-lindhorst.de
 */


package lindhorst.apps.jdbc.swing.administration;

import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import lindhorst.apps.jdbc.swing.data.*;
import lindhorst.apps.jdbc.swing.modules.*;
import lindhorst.apps.jdbc.swing.helpers.*;

public class TabbedModuleView extends JPanel
{
  private JTabbedPane pane=new JTabbedPane(SwingConstants.TOP);
  private SessionDataModel model=null;

  public TabbedModuleView(SessionDataModel model)
  {
    this.model=model;
    setLayout(new BorderLayout());
    pane.addTab(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("tables"),new ModuleView(Module.TABLE));
    pane.addTab(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("queries"),new ModuleView(Module.QUERY));
    add(pane,BorderLayout.CENTER);
  }


  protected void openModule(String moduleType, String moduleName)
  {
        model.requestModule(moduleType,moduleName);
  }

  protected void eraseModule(String moduleType, String moduleName)
  {
    Properties props=model.getProperties();
    boolean found=false;
    int index=0;
    String prefix="jdbcsession."+moduleType+".";
    String suffix=".name";
    String prop=null;

    for(index=0;(prop=props.getProperty(prefix+index+suffix))!=null;index++)
    {
      if(prop.equals(moduleName))
      {
        found=true;
        break;
      }
    }

    //found?
    if(!found)
    {
      JOptionPane.showMessageDialog(Helpers.getTopLevelContainer(),java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("modul")+moduleName+java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("not_found"),java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("not_found.title"),JOptionPane.ERROR_MESSAGE);
      return;
    }

    //shift following modules
    for(;(prop=props.getProperty(prefix+(index+1)+suffix))!=null;index++)
      props.put(prefix+index+suffix,prop);

    //remove last module
    props.remove(prefix+index+suffix);

    //remove description and command
    props.remove("jdbcsession."+moduleType+"."+moduleName+".command");
    props.remove("jdbcsession."+moduleType+"."+moduleName+".description");

    try
    {
      model.save();
    }
    catch(Exception e)
    {
      Helpers.showError(e);
    }
  }

  private class ModuleView extends JPanel implements ListSelectionListener, ActionListener, ChangeListener
  {
    private JList list;
    private String moduleType;
    private JButton open, create, erase;

    ModuleView(String moduleType)
    {
      this.moduleType=moduleType;

      list=new JList();
      list.setCellRenderer(new ModuleListCellRenderer());
      list.addListSelectionListener(this);
      JScrollPane scrollPane=new JScrollPane(list);

      JPanel box=new JPanel(new FlowLayout());
      JPanel buttonTemp=new JPanel(new BorderLayout(10,10));
      open=new JButton(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("open"));
      open.setMnemonic(0);
      open.setEnabled(false);
      open.setToolTipText(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("open.long"));
      open.addActionListener(this);
      buttonTemp.add(open,BorderLayout.NORTH);
      create=new JButton(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("new"));
      create.setMnemonic(0);
      create.setToolTipText(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("new.long"));
      create.addActionListener(this);
      buttonTemp.add(create,BorderLayout.CENTER);
      erase=new JButton(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("delete"));
      erase.setEnabled(false);
      erase.setMnemonic(0);
      erase.setToolTipText(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("delete.long"));
      erase.addActionListener(this);
      buttonTemp.add(erase,BorderLayout.SOUTH);
      box.add(buttonTemp);

      setLayout(new BorderLayout());
      add(scrollPane,BorderLayout.CENTER);
      add(box,BorderLayout.EAST);
      model.addChangeListener(this);
    }

    public void actionPerformed(ActionEvent ae)
    {
      if(ae.getActionCommand().equals(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("new")))
      {
        model.requestModule(moduleType,"<Neu>");
        return;
      }
      if(ae.getActionCommand().equals(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("open")))
      {
        String[] pair=(String[])list.getSelectedValue();
        openModule(moduleType,pair[0]);
        list.clearSelection();
        open.setEnabled(false);
        erase.setEnabled(false);
        return;
      }
      if(ae.getActionCommand().equals(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("delete")))
      {
        String[] pair=(String[]) list.getSelectedValue();
        eraseModule(moduleType,pair[0]);
        //update done by SessionDataModel

        list.clearSelection();
        open.setEnabled(false);
        erase.setEnabled(false);
      }
    }

    public void valueChanged(ListSelectionEvent event)
    {
      if(event.getValueIsAdjusting()) return;
      open.setEnabled(true);
      erase.setEnabled(true);
    }

    public void stateChanged(ChangeEvent event)
    {
      extractModuleNames();
      list.setListData(extractModuleNames());
      list.clearSelection();
      open.setEnabled(false);
      erase.setEnabled(false);
    }

    private Object[] extractModuleNames()
    {
      Vector vector=new Vector(100);

      Properties props=model.getProperties();
      String prefix="jdbcsession."+moduleType+".";
      String suffix=".name";
      String name,description;
      String[] pair;
      int index=0;

      while((name=props.getProperty(prefix+index+suffix))!=null)
      {
        pair=new String[2];
        pair[0]=name;
        pair[1]=props.getProperty(prefix+name+".description");
        vector.addElement(pair);
        index++;
      }

      vector.trimToSize();

      Object[] temp=new Object[vector.size()];
      vector.copyInto(temp);

      sort(temp);

      return temp;
    }

    private final void sort(Object[] array)
    {
      int index=0;
      String name1,name2;
      Object temp;

      for(int i=0;i<array.length-1;i++)
      {
        index=i; //smallest Element;
        name1=((String[])array[i])[0].toLowerCase();

        for(int j=i+1;j<array.length;j++)
        {
          if((name2=((String[])array[j])[0].toLowerCase()).compareTo(name1)<0)
          {
            name1=name2;
            index=j;
          }
        }

        if(index!=i)
        {
          temp=array[i];
          array[i]=array[index];
          array[index]=temp;
        }
      }
    }
  }
}

class ModuleListCellRenderer implements ListCellRenderer
{
  public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean hasFocus)
  {
    String[] pair=(String[])value;
    
    JPanel panel=new JPanel(new ListCellRendererSuperLayout());
    if(isSelected)
      panel.setBackground(SystemColor.textHighlight);
    else
      panel.setBackground(list.getBackground());
    panel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));

    JLabel lab;
    for(int i=0;i<pair.length;i++)
    {
      lab=new JLabel(pair[i]);
      lab.setBorder(BorderFactory.createEmptyBorder(2,3,6,2));
      if(isSelected)
        lab.setForeground(SystemColor.textHighlightText);
      panel.add(lab);
    }
    return panel;
  }
}

/**
*Organizes all ListCellRenderers for the view in columns
*/
class ListCellRendererSuperLayout implements LayoutManager
{
  /**holds preferred widths per column*/
  private static int[] width=new int[2];
  static
  {
    for(int i=0;i<width.length;i++)
      width[i]=0;
  }
  /**holds preferred height*/
  private static int height=0;
  /**number of columns*/
  private int columns=2;


  /**
  *Does nothing.
  */
  public void addLayoutComponent(String name, Component component)
  {
    //nothing to do
  }

  /**
  *Does nothing.
  */
  public void removeLayoutComponent(Component component)
  {
    //nothing to do
  }

  /**
  *Finds the preferred layout size for the container.
  *@param c the container to be layed out
  */
  public Dimension preferredLayoutSize(Container c)
  {
    Component[] comps=c.getComponents();
    if(comps==null || comps.length==0) return new Dimension(0,0);

    //Find max preferred height
    int size;
    for(int i=0;i<comps.length;i++)
    {
      size=comps[i].getPreferredSize().height;
      if(size>height) height=size;
    }

    //Find preferred width per column
    int index=0;
    while(index<comps.length)
    {
      for(int lineIndex=0;lineIndex<columns && index<comps.length;lineIndex++,index++)
      {
        size=comps[index].getPreferredSize().width;
        if(size>width[lineIndex])
          width[lineIndex]=size;
      }
    }

    //calculate totals
    boolean overflow=comps.length%columns>0;
    int rows=comps.length/columns;
    if(overflow) rows++;
    int totalHeight=rows*height;
    int totalWidth=0;
    for(int i=0;i<columns;i++)
      totalWidth+=width[i];

    //watch insets!
    Insets insets=c.getInsets();
    totalWidth=totalWidth+insets.left+insets.right;
    totalHeight=totalHeight+insets.top+insets.bottom;

    return new Dimension(totalWidth,totalHeight);
  }

  public void layoutContainer(Container c)
  {
    Component[] comps=c.getComponents();
    if(comps==null || comps.length==0) return;

    Dimension pSize=preferredLayoutSize(c);
    //watch insets!
    Insets insets=c.getInsets();
    pSize=new Dimension(pSize.width-insets.left-insets.right,pSize.height-insets.top-insets.bottom);

    //enough space?
    boolean overflow=comps.length%columns>0;
    int rows=comps.length/columns;
    if(overflow) rows++;

    int containerHeight=c.getSize().height-insets.top-insets.bottom;
    int actualHeight=height;       //set by preferredLayoutSize
    if(containerHeight<pSize.height)
      actualHeight=containerHeight/rows;

    //check width
    int containerWidth=c.getSize().width-insets.left-insets.right;
    boolean useCalculatedWidths=containerWidth>=pSize.width;

    if(useCalculatedWidths)
    {
      int index=0;      //for component array
      int x=insets.left;          //x coordinate
      int y=insets.top;          //y coordinate
      while(index<comps.length)
      {
        x=insets.left;

        for(int lineIndex=0;lineIndex<columns && index<comps.length;lineIndex++,index++)
        {
          comps[index].setBounds(x,y,width[lineIndex],actualHeight);
          x+=width[lineIndex];
        }

        y+=actualHeight;
      }
    }
    else
    {
      int index=0;
      int x=insets.left;
      int y=insets.top;
      int actualWidth=containerWidth/columns;
      while(index<comps.length)
      {
        x=insets.top;

        for(int lineIndex=0;lineIndex<columns && index<comps.length;lineIndex++,index++)
        {
          comps[index].setBounds(x,y,actualWidth,actualHeight);
          x+=width[lineIndex];
        }

        y+=actualHeight;
      }
    }
  }

  /**
  *Finds the minimum layout size for the container.
  *@param c the container to be layed out
  */
  public Dimension minimumLayoutSize(Container c)
  {
    return preferredLayoutSize(c);
  }
}
