/* 
 * This is JavaSQL, a tool to check the validity of connections and queries
 * in JDBC.
 * Copyright (C) 2002 Alexander Lindhorst
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact:
 * Alexander Lindhorst
 * Elsa-Brandstrm-Weg 3
 * 33102 Paderborn
 * Germany
 * al@alexander-lindhorst.de
 */

package lindhorst.apps.jdbc.swing.data;

import java.sql.*;
import java.util.*;
import java.io.*;
import java.net.*;
import javax.swing.*;
import javax.swing.table.*;
import javax.swing.event.*;
import lindhorst.apps.jdbc.swing.helpers.*;
import lindhorst.apps.jdbc.swing.info.*;
import lindhorst.apps.jdbc.swing.*;
import lindhorst.apps.jdbc.swing.administration.*;
import lindhorst.apps.jdbc.swing.event.*;
import lindhorst.apps.jdbc.swing.sql.*;

public class SessionDataModel {
    public static final String NO_MODULE_NAME=ModuleRequestEvent.NO_NAME;
    
    //internal constants
    private static final int READ_CONNECTION=0;
    private static final int WRITE_CONNECTION=1;
    
    private Connection readConnection=null;
    private Connection writeConnection=null;
    private SQLException exception=null;
    private Properties props=null;
    private StringBuffer warnings=new StringBuffer();
    private EventListenerList listeners=new EventListenerList();
    
    
    public SessionDataModel(Properties props) throws SQLException, ClassNotFoundException {
        this(props,null);
    }
    
    public SessionDataModel(Properties props,LoginListener listener) throws SQLException, ClassNotFoundException {
        this.props=props;
        addLoginListener(listener);
        init();
        System.gc();
        System.runFinalization();
    }
    
    public SessionDataModel() throws SQLException,ClassNotFoundException {
        this(null);
    }
    
    public void init() throws SQLException, ClassNotFoundException {
        
        SQLException exception=null;
        ConnectionFetcher readFetcher=new ConnectionFetcher(READ_CONNECTION);
        ConnectionFetcher writeFetcher=new ConnectionFetcher(WRITE_CONNECTION);
        
        readFetcher.start();
        writeFetcher.start();
    }
    
    public void close() throws SQLException {
        fireLoginStatusChanged(LoginEvent.LOGOUT_EVENT);
        try {
            if(readConnection!=null)readConnection.close();
            if(writeConnection!=null)writeConnection.close();
        }
        catch(Exception e) {
            e.printStackTrace();
        }
        fireLoginStatusChanged(LoginEvent.LOGOUT_COMPLETED_EVENT);
    }
    
    public void finalize() throws Throwable {
        close();
    }
    
    
    public Properties getProperties() {
        return props;
    }
    
    public Connection getReadableConnection() {
        return readConnection;
    }
    
    public Connection getWritableConnection() {
        return writeConnection;
    }
    
    public SQLException getRaisedException() {
        SQLException localCopy=exception;
        exception=null; //deliver only once
        return localCopy;
    }
    
    public ModuleAdministration getAdministrationPanel() {
        return new ModuleAdministration(this);
    }
    
    public void requestModule(String moduleType, String moduleName) {
        fireModuleRequestEvent(moduleType, moduleName);
    }
    
    public void addLoginListener(LoginListener listener) {
        listeners.add(LoginListener.class,listener);
    }
    
    public void removeLoginListener(LoginListener listener) {
        listeners.remove(LoginListener.class,listener);
    }
    
    private void fireLoginStatusChanged(int type) {
        EventListener[] array=listeners.getListeners(LoginListener.class);
        LoginEvent event=new LoginEvent(this,type);
        for (int i=0;i<array.length;i++) {
            ((LoginListener)array[i]).loginStatusChanged(event);
        }
    }
    
    public void addChangeListener(ChangeListener listener) {
        if(listener==null) return;
        listeners.add(ChangeListener.class,listener);
        listener.stateChanged(new ChangeEvent(this));
    }
    
    public void removeChangeListener(ChangeListener listener) {
        listeners.remove(ChangeListener.class,listener);
    }
    
    private void fireChangeEvent() {
        ChangeEvent event=new ChangeEvent(this);
        Object[] changeListeners=listeners.getListenerList();
        
        for(int i=0;i<changeListeners.length;i+=2) {
            if(changeListeners[i]==ChangeListener.class)
                ((ChangeListener)changeListeners[i+1]).stateChanged(event);
        }
    }
    
    public void addModuleRequestListener(ModuleRequestListener listener) {
        listeners.add(ModuleRequestListener.class,listener);
    }
    
    public void removeModuleRequestListener(ModuleRequestListener listener) {
        listeners.remove(ModuleRequestListener.class,listener);
    }
    
    private void fireModuleRequestEvent(String type, String name) {
        ModuleRequestEvent event=new ModuleRequestEvent(this,type,name);
        Object[] list=listeners.getListenerList();
        
        for(int i=0;i<list.length;i+=2)
            if(list[i]==ModuleRequestListener.class)
                ((ModuleRequestListener)list[i+1]).moduleRequested(event);
    }
    
    public void save() throws IOException {
        String fileName=props.getProperty("jdbcsession.file");
        System.out.println(fileName);
        
        //Password rausnehmen
        props.remove("jdbcsession.password");
        
        try {
            File file=new File(fileName);
            if(file.exists()) file.delete();
            
            FileOutputStream stream=new FileOutputStream(props.getProperty("jdbcsession.file"));
            props.store(stream,"Do not modify - this file is machine created!");
            fireChangeEvent();
        }
        catch(IOException exc) {
            throw new IOException("Konnte Session nicht speichern - Datei nicht gefunden!");
        }
    }
    
    private static URL[] getURLArray(String concatenatedPath) {
        ArrayList list=new ArrayList(10);
        StringTokenizer tokenizer=new StringTokenizer(concatenatedPath,File.pathSeparator,false);
        File file=null;
        while(tokenizer.hasMoreTokens()) {
            file=new File(tokenizer.nextToken());
            try {
                list.add(file.toURL());
            }
            catch(Exception e) {
                e.printStackTrace(System.err);
            }
        }
        Object[] temp=list.toArray();
        URL[] array=new URL[temp.length];
        System.arraycopy(temp,0,array,0,temp.length);
        
        return array;
    }
    
    /**
     *This separate Thread is needed for the GUI not to block
     *while attempting to establish a connection to the database
     *system. It will then start another thread with its own class loader, so
     *the JDBC DriverManager can work with that classloader.
     */
    
    private class ConnectionFetcher extends Thread {
        private Connection connection=null;
        private int connectionType=-1;
        
        ConnectionFetcher(int connectionType) {
            this.connectionType=connectionType;
        }
        
        public void run() {
            try{
                SwingUtilities.invokeAndWait(new LoginEventFirer(LoginEvent.LOGIN_EVENT));
            }
            catch(Exception ex) {
                ex.printStackTrace(System.err);
            }
            Thread.yield(); //let EventDispatchThread do its work
            
            //form ClassLoader information
            String path=props.getProperty("jdbcsession.classpathextension");
            ClassLoader loader=null;
            if(path!=null && path.length()>0) {
                URL[] urls=getURLArray(path);
                loader=new URLClassLoader(urls);
            }
            else {
                loader=getClass().getClassLoader();
            }
            
            //give other threads a sec to clean up
            String url=props.getProperty("jdbcsession.url");
            String driver=props.getProperty("jdbcsession.driver");
            String user=props.getProperty("jdbcsession.user");
            String password=props.getProperty("jdbcsession.password");
            
            try {
                Driver d=new UniversalDriverWrapper(loader,driver);
                if(user==null||password==null) {
                    connection=DriverManager.getConnection(url);
                }
                else {
                    connection=DriverManager.getConnection(url,user,password);
                }
            }
            catch(Exception e) {
                if(e instanceof SQLException)
                    exception=(SQLException)e;
                else
                    exception=new SQLException("SQLException based on: "+e.toString());
                SwingUtilities.invokeLater(new LoginEventFirer(LoginEvent.LOGIN_COMPLETED_EVENT));
            }
            switch(connectionType) {
                case READ_CONNECTION:
                    readConnection=connection;
                    break;
                case WRITE_CONNECTION:
                    writeConnection=connection;
                    break;
                default:
                    exception=new SQLException("Not a valid connection type");
            }
            SwingUtilities.invokeLater(new LoginEventFirer(LoginEvent.LOGIN_COMPLETED_EVENT));
        }
        
    }
    
    private class LoginEventFirer implements Runnable {
        private int type=-1;
        
        LoginEventFirer(int type) {
            this.type=type;
        }
        
        /** When an object implementing interface <code>Runnable</code> is used
         * to create a thread, starting the thread causes the object's
         * <code>run</code> method to be called in that separately executing
         * thread.
         * <p>
         * The general contract of the method <code>run</code> is that it may
         * take any action whatsoever.
         *
         * @see     java.lang.Thread#run()
         */
        public void run() {
            fireLoginStatusChanged(type);
        }
        
    }
}
