/* 
 * This is JavaSQL, a tool to check the validity of connections and queries
 * in JDBC.
 * Copyright (C) 2002 Alexander Lindhorst
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact:
 * Alexander Lindhorst
 * Elsa-Brandstrm-Weg 3
 * 33102 Paderborn
 * Germany
 * al@alexander-lindhorst.de
 */


package lindhorst.apps.jdbc.swing.helpers;

import javax.swing.*;
import java.awt.*;
import java.io.*;
import java.util.*;
import java.text.*;
import lindhorst.swing.filechooser.ExtensionFileFilter;

public class Helpers {
    private static boolean errorBeingShown=false;
    private static Frame topLevel=null;
    
    /**
     *Centers the given window with respects to the top level frame. If there is no such
     *frame or it's not visible at the moment, the window will be centered on
     *the desktop.
     */
    public static void doCenterWindow(Window w) {
        Toolkit tk=Toolkit.getDefaultToolkit();
        Dimension size=w.getSize();
        Point location=null;
        if(topLevel==null||!topLevel.isShowing()||topLevel==w) {
            Dimension screen=tk.getScreenSize();
            location=new Point((screen.width-size.width)/2,(screen.height-size.height)/2);
        }
        else {
            Point topLocation=topLevel.getLocationOnScreen();
            Dimension topSize=topLevel.getSize();
            int xOffset=(topSize.width-size.width)/2;
            int yOffset=(topSize.height-size.height)/2;
            location=new Point(topLocation.x+xOffset,topLocation.y+yOffset);
        }
        w.setLocation(location);
    }
    
    public static void setTopLevelContainer(Frame container) {
        topLevel=container;
    }
    
    public static Frame getTopLevelContainer() {
        return topLevel;
    }
    
    public static void toFront() {
        if(topLevel!=null && topLevel instanceof Window) {
            ((Window)topLevel).toFront();
            topLevel.requestFocus();
        }
    }
    
    public static void showError(Throwable t) {
        if(errorBeingShown) {
            return;
        }
        errorBeingShown=true;
        System.err.println(t.getLocalizedMessage());
        if(t.getMessage()!=null)
            JOptionPane.showMessageDialog((Component)topLevel,t.getMessage(),t.getClass().getName(),JOptionPane.ERROR_MESSAGE);
        else
            JOptionPane.showMessageDialog((Component)topLevel,t.getClass().getName(),t.getClass().getName(),JOptionPane.ERROR_MESSAGE);
        errorBeingShown=false;
    }
    
    public static void showErrorInternally(Throwable t, Component parent) {
        if(errorBeingShown) {
            return;
        }
        errorBeingShown=true;
        if(t.getMessage()!=null)
            JOptionPane.showInternalMessageDialog(parent,t.getMessage(),t.getClass().getName(),JOptionPane.ERROR_MESSAGE);
        else
            JOptionPane.showInternalMessageDialog(parent,t.getClass().getName(),t.getClass().getName(),JOptionPane.ERROR_MESSAGE);
        errorBeingShown=false;
        t.printStackTrace(System.err);
    }
    
    public static void showInfo(String info) {
        JOptionPane.showMessageDialog((Component)topLevel,info,"Information",JOptionPane.INFORMATION_MESSAGE);
    }
    
    public static byte[] getResource(Class c, String fileName) {
        byte[] bytes=null;
        
        try {
            InputStream in=c.getResourceAsStream(fileName);
            ByteArrayOutputStream out=new ByteArrayOutputStream(4096);
            int b;
            while( (b=in.read())!=-1) out.write(b);
            bytes=out.toByteArray();
        }
        catch(Exception e) {
        }
        
        return bytes;
    }
    
    public static Image getImage(Class c, String fileName) {
        return Toolkit.getDefaultToolkit().createImage(getResource(c,fileName));
    }
    
    public static ImageIcon getIcon(Class c, String fileName) {
        return new ImageIcon(getResource(c,fileName));
    }
    
    public static File showOpenFileDialog() {
        JFileChooser chooser=new JFileChooser(System.getProperty("user.dir"));
        ExtensionFileFilter filter=new ExtensionFileFilter();
        filter.addExtension("properties");
        filter.setDescription("Java-Properties Dateien");
        chooser.setFileFilter(filter);
        
        int retVal=chooser.showOpenDialog(topLevel);
        if(retVal==chooser.APPROVE_OPTION) return chooser.getSelectedFile();
        else return null;
    }
    
    public static File showSaveFileDialog() {
        JFileChooser chooser=new JFileChooser(System.getProperty("user.dir"));
        ExtensionFileFilter filter=new ExtensionFileFilter();
        filter.addExtension("properties");
        filter.setDescription("Java-Properties Dateien");
        chooser.setFileFilter(filter);
        
        int retVal=chooser.showSaveDialog(topLevel);
        if(retVal==chooser.APPROVE_OPTION) return chooser.getSelectedFile();
        else return null;
    }
    
    
    public static void setCursor(Component comp, Cursor cursor) {
        
        if(comp instanceof Container) {
            Component[] comps=((Container)comp).getComponents();
            for(int i=0;i<comps.length;i++)
                setCursor(comps[i],cursor);
        }
        comp.setCursor(cursor);
    }
    
    public static Number convertToNumber(String from, Class toClass) {
        if(toClass==Double.class || toClass==Float.class) {
            Double d=Double.valueOf(from);
            if(toClass==Double.class) return d;
            else {
                return new Float(d.floatValue());
            }
        }
        else {
            if(toClass==Long.class) return new Long(from);
        }
        
        return new Integer(Double.valueOf(from).intValue());
    }
    
    public static Date convertToDate(String from, Locale locale) {
        Date d=null;
        DateFormat formatter=DateFormat.getDateInstance(DateFormat.SHORT,locale);
        d=parseDate(formatter,from);
        if(d!=null) return d;
        else formatter=DateFormat.getDateInstance(DateFormat.MEDIUM,locale);
        d=parseDate(formatter,from);
        if(d!=null) return d;
        else formatter=DateFormat.getDateInstance(DateFormat.LONG,locale);
        d=parseDate(formatter,from);
        if(d!=null)return d;
        else formatter=DateFormat.getDateTimeInstance(DateFormat.LONG,DateFormat.LONG,locale);
        d=parseDate(formatter,from);
        if(d!=null) return d;
        //didn't work with default locale
        //try it with a neutral instance
        else formatter=DateFormat.getInstance();
        d=parseDate(formatter,from);
        if(d!=null) return d;
        //try it with US Locale
        if(!locale.equals(Locale.US)) {
            //System.out.println("Trying US locale ...");
            d=convertToDate(from,Locale.US);
        }
        if(d!=null) return d;
        else {
            //System.out.println("Trying JDBC date types ...");
            d=parseSQLDate(from);
        }
        
        return d;
    }
    
    private static Date parseDate(DateFormat format,String date) {
        Date d=null;
        try {
            d=format.parse(date);
        }
        catch(Exception exc) {
        }
        
        return d;
    }
    
    private static Date parseSQLDate(String date) {
        Date d=null;
        
        try {
            d=java.sql.Timestamp.valueOf(date);
        }
        catch(Exception e) {
        }
        if(d!=null) return d;
        
        try {
            d=java.sql.Date.valueOf(date);
        }
        catch(Exception e) {
        }
        if(d!=null) return d;
        
        try {
            d=java.sql.Time.valueOf(date);
        }
        catch(Exception e) {
        }
        return d;
    }
}
