/* 
 * This is JavaSQL, a tool to check the validity of connections and queries
 * in JDBC.
 * Copyright (C) 2002 Alexander Lindhorst
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact:
 * Alexander Lindhorst
 * Elsa-Brandstrm-Weg 3
 * 33102 Paderborn
 * Germany
 * al@alexander-lindhorst.de
 */


package lindhorst.apps.jdbc.swing.modules;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import java.awt.*;
import java.awt.event.*;
import java.sql.*;
import java.util.*;
import lindhorst.apps.jdbc.swing.*;
import lindhorst.apps.jdbc.swing.helpers.*;
import lindhorst.apps.jdbc.swing.modules.datamodels.*;

public class Table extends JInternalFrame implements Module
{
  private TableDataModel model=null;
  private JTable table=null;
  private JLabel infoLabel;
  private JProgressBar bar;
  private JPanel info;
  private JButton deleter;

  public Table(TableDataModel model)
  {
    super(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("table"),true,true,true,true);
    this.model=model;
    setFrameIcon(Helpers.getIcon(getClass(),"tables.gif"));
    setTitle(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("table")+model.getTableName());
    setOpaque(false);
    init();
    pack();
    setVisible(true);
  }

  public Table(Connection readableConnection,Connection writableConnection,String name)
  {
    this(new TableDataModel(readableConnection,writableConnection,name));
  }

  protected void init()
  {

    table=new JTable();
    table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
    JTableHeader header=table.getTableHeader();
    header.addMouseListener(new MouseAdapter()
    {
      public void mouseClicked(MouseEvent event)
      {
        JTableHeader header=(JTableHeader)event.getSource();
        int x=event.getPoint().x;
        TableColumnModel columnModel=header.getColumnModel();
        TableColumn column=columnModel.getColumn(columnModel.getColumnIndexAtX(x));
        String columnName=column.getHeaderValue().toString();
        model.doSort(columnName);
      }
    });
    JScrollPane tableView=new JScrollPane(table);
    tableView.setPreferredSize(new Dimension(400,250));

    JPanel cPane=new JPanel(new GridLayout(1,1));

    cPane.add(tableView);
    cPane.setBorder(BorderFactory.createTitledBorder(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("table")));

    info=new JPanel(new GridLayout(1,2));
    infoLabel=new JLabel(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("reading_table:"));
    info.add(infoLabel);
    bar=new JProgressBar(0,100);
    info.add(bar);
    info.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEtchedBorder(),BorderFactory.createEmptyBorder(5,5,5,5)));
    model.addChangeListener(new BarUpdater());

    JPanel buttons=new JPanel(new GridLayout(0,1));
    JButton button=new JButton(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("save"));
    button.setMnemonic('S');
    buttons.add(button);
    button.addActionListener(new ActionListener()
    {
      public void actionPerformed(ActionEvent event)
      {
        save();
      }
    });
    deleter=new JButton(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("delete_row"));
    deleter.setActionCommand("DELETE");
    deleter.setMnemonic('l');
    deleter.setEnabled(false);
    deleter.addActionListener(new LineDeleter());
    buttons.add(deleter);
    button=new JButton(java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("insert_row"));
    button.setMnemonic('e');
    button.addActionListener(new LineInserter());
    buttons.add(button);
    JPanel outerButtons=new JPanel(new FlowLayout(FlowLayout.CENTER));
    outerButtons.add(buttons);

    JPanel all=new JPanel(new BorderLayout(5,5));
    all.add(cPane,BorderLayout.CENTER);
    all.add(outerButtons,BorderLayout.EAST);
    all.add(info,BorderLayout.SOUTH);

    setContentPane(all);

    //plug into model
    table.setModel(model);
    table.getSelectionModel().addListSelectionListener(new DeleteEnabler());
    
    //change cursor when necessary
    model.addChangeListener(new Table.CursorSwitch());
  }

  public void save()
  {
    String name=model.getTableName();

    Object[] names=extractTableNames();
    boolean doAgain=true;

    //get description from user
    String description=JOptionPane.showInputDialog(Helpers.getTopLevelContainer(),java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("description"),java.util.ResourceBundle.getBundle("lindhorst/apps/jdbc/swing/i18n/administration").getString("save_as"),JOptionPane.QUESTION_MESSAGE);
    if(description==null) return; //canceled

    //Test: do we already have that query?
    boolean alreadyThere=false;
    for(int i=0;i<names.length;i++)
    {
      if(((String)names[i]).equals(name))
      {
        alreadyThere=true;
        break;
      }
    }

    Properties props=((AppPane)getParent()).getModel().getProperties();

    //if no: save a new index!
    if(!alreadyThere)
      props.put("jdbcsession."+TABLE+"."+names.length+".name",name);

    //save command and description
    props.put("jdbcsession."+TABLE+"."+name+".description",description);

    //save props
    try
    {
      ((AppPane)getParent()).getModel().save();
    }
    catch(Exception exc)
    {
      Helpers.showError(exc);
    }
  }

  private Object[] extractTableNames()
  {
    ArrayList array=new ArrayList(20);
    Properties props=((AppPane)getParent()).getModel().getProperties();
    int index=0;
    String name;

    while((name=props.getProperty("jdbcsession."+TABLE+"."+index+".name"))!=null)
    {
      array.add(name);
      index++;
    }

    return array.toArray();
  }

  private JInternalFrame getInstance()
  {
    return this;
  }

  private class BarUpdater implements ChangeListener
  {
    public void stateChanged(ChangeEvent event)
    {
      bar.setMaximum(model.getNumberOfRecords());
      bar.setValue(model.getNumberOfRecordsRead());
      if(model.getNumberOfRecordsRead()<=0||bar.getPercentComplete()<1.0)
      {
        bar.setVisible(true);
        if(model.isAssessingTableSize()) {
            infoLabel.setText("Schtze Tabellengre ...");
        }
        else {
            infoLabel.setText("Lese Tabellendaten ...");
        }
        infoLabel.setVisible(true);
      }
      else
      {
        bar.setVisible(false);
        infoLabel.setVisible(false);
        info.repaint(0L);
      }
    }
  }
  
  private class CursorSwitch implements ChangeListener {
      
      /** Invoked when the target of the listener has changed its state.
       *
       * @param e  a ChangeEvent object
       */
      public void stateChanged(ChangeEvent e) {
          if(model!=null && model.isRetrievingData()) {
              Helpers.setCursor(getInstance(),Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
          }
          else {
              Helpers.setCursor(getInstance(),Cursor.getDefaultCursor());
          }
      }      
  }

  private class DeleteEnabler implements ListSelectionListener
  {
    public void valueChanged(ListSelectionEvent event)
    {
      deleter.setEnabled(!((ListSelectionModel)event.getSource()).isSelectionEmpty());
    }
  }

  private class LineDeleter implements ActionListener
  {
    public void actionPerformed(ActionEvent ae)
    {
      model.deleteLines(table.getSelectedRows());
    }
  }

  private class LineInserter implements ActionListener
  {
    public void actionPerformed(ActionEvent ae)
    {
      SingleRecordEditor editor=new SingleRecordEditor(model.getColumnNames(),model.getColumnClasses());
      if(editor.isProperlyClosed())
      {
        try
        {
          model.insertLine(editor.getData());
        }
        catch(SQLException e)
        {
          Helpers.showError(e);
        }
      }
    }
  }
}
