/* 
 * This is JavaSQL, a tool to check the validity of connections and queries
 * in JDBC.
 * Copyright (C) 2002 Alexander Lindhorst
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact:
 * Alexander Lindhorst
 * Elsa-Brandstrm-Weg 3
 * 33102 Paderborn
 * Germany
 * al@alexander-lindhorst.de
 */

package lindhorst.apps.jdbc.swing.modules.datamodels;

import java.sql.*;
import java.util.*;
import javax.swing.*;
import javax.swing.table.*;
import javax.swing.event.*;
import lindhorst.apps.jdbc.swing.helpers.*;

public class QueriesDataModel {
    private Connection readConnection=null;
    private Connection writeConnection=null;
    private StringBuffer warnings=new StringBuffer();
    private EventListenerList listeners=new EventListenerList();
    private Vector oldCommands=new Vector(10,10);
    private Object[][] results=new String[0][0];
    private String[] heads=new String[0];
    private Class[] colClasses=null;
    private boolean executing=false;
    
    public QueriesDataModel(Connection readableConnection,Connection writableConnection) {
        this.readConnection=readableConnection;
        this.writeConnection=writableConnection;
    }
    
    
    public void executeSQL(String sql) {
        new SQLExecuter(sql).start();
    }
    
    private synchronized void process(String sql) {
        warnings=new StringBuffer();
        results=new String[0][0];
        executing=true;
        fireChangeEvent();
        oldCommands.addElement(sql);
        Connection connection=null;
        
        //what kind?
        int pos=sql.indexOf(" ");
        
        try {
            if(pos<0) throw new SQLException("Not a valid sql command!");
            String command=sql.substring(0,pos);
            if(command.equalsIgnoreCase("select"))
                connection=readConnection;
            else
                connection=writeConnection;
            
            connection.clearWarnings();
            connection.setAutoCommit(true);
            long start=System.currentTimeMillis();
            Statement stmt=connection.createStatement();
            boolean executed=stmt.execute(sql);
            
            if(executed) {
                ResultSet rs=stmt.getResultSet();
                if(rs!=null) {
                    
                    ResultSetMetaData rmd=rs.getMetaData();
                    int size=rmd.getColumnCount();
                    heads=new String[size];
                    colClasses=new Class[size];
                    ArrayList array=new ArrayList();
                    
                    for(int i=1;i<=size;i++) {
                        heads[i-1]=rmd.getColumnName(i);
                        colClasses[i-1]=getClassForColumn(rmd.getColumnType(i));
                    }
                    
                    Object[] data;
                    String item;
                    
                    while(rs.next()) {
                        data=new Object[size];
                        
                        for(int i=1;i<=size;i++) {
                            data[i-1]=rs.getObject(i);
                        }
                        array.add(data);
                    }
                    
                    results=new Object[array.size()][];
                    Object temp=array.toArray();
                    System.arraycopy(temp,0,results,0,results.length);
                    
                }
                
                int count=stmt.getUpdateCount();
                if(count>=0)
                    Helpers.showInfo(count+" Datenstze gendert!");
                
                long end=System.currentTimeMillis();
                rs.close();
                stmt.close();
            }
            
            //Warnings auslesen
            SQLWarning warning=connection.getWarnings();
            String state=null;
            while(warning!=null) {
                state=warning.getSQLState();
                if(state!=null) {
                    warnings.append("SQL Status: ");
                    warnings.append(warning.getSQLState());
                    warnings.append('\n');
                    state=null;
                }
                warnings.append(warning.getMessage());
                warnings.append('\n');
                warning=warning.getNextWarning();
            }
        }
        catch(Exception e) {
            if(e instanceof SQLException) {
                String state=null;
                String[] tmp={"Fehler"};
                heads=tmp;
                ArrayList array=new ArrayList(10);
                while(e!=null) {
                    state=((SQLException)e).getSQLState();
                    if(state!=null) {
                        warnings.append("SQL Status: ");
                        warnings.append(state);
                        warnings.append('\n');
                        state=null;
                    }
                    warnings.append(e.getMessage());
                    e=((SQLException)e).getNextException();
                }
            }
            else
                Helpers.showError(e);
        }
        finally {
            executing=false;
            fireChangeEvent();
        }
    }
    
    public String getWarnings() {
        return warnings.toString();
    }
    
    public Vector getOldCommands() {
        return oldCommands;
    }
    
    public boolean isExecuting() {
        return executing;
    }
    
    public TableModel getTableModelFromOutput() {
        return new TableModelRepresentation();
    }
    
    public void addChangeListener(ChangeListener listener) {
        listeners.add(ChangeListener.class,listener);
        listener.stateChanged(new ChangeEvent(this));
    }
    
    public void removeChangeListener(ChangeListener listener) {
        listeners.remove(ChangeListener.class,listener);
    }
    
    private void fireChangeEvent() {
        final ChangeEvent event=new ChangeEvent(this);
        final EventListener[] array=listeners.getListeners(ChangeListener.class);
        if(SwingUtilities.isEventDispatchThread()) {
            for(int i=0;i<array.length;i++) {
                ( (ChangeListener)array[i] ).stateChanged(event);
            }
        }
        else{
            SwingUtilities.invokeLater(new Runnable(){
                public void run() {
                    for(int i=0;i<array.length;i++) {
                        ((ChangeListener)array[i]).stateChanged(event);
                    }
                }
            });
        }
    }
    
    
    private class SQLExecuter extends Thread {
        private String sql;
        
        SQLExecuter(String sql) {
            this.sql=sql;
            setPriority(Thread.NORM_PRIORITY+1);
        }
        
        public void run() {
            Thread.yield();
            process(sql);
        }
    }
    
    private static Class getClassForColumn(int type) {
        switch(type) {
            case Types.BIGINT: return Number.class;
            case Types.BIT: return Boolean.class;
            case Types.CHAR:  return String.class;
            case Types.DATE:  return java.util.Date.class;
            case Types.DECIMAL: return Double.class;
            case Types.DOUBLE:  return Double.class;
            case Types.FLOAT: return Float.class;
            case Types.INTEGER: return Integer.class;
            case Types.LONGVARCHAR: return String.class;
            case Types.NUMERIC: return Number.class;
            case Types.REAL:  return Number.class;
            case Types.SMALLINT: return Integer.class;
            case Types.TIME:  return java.util.Date.class;
            case Types.TIMESTAMP: return java.util.Date.class;
            case Types.TINYINT: return Integer.class;
            
            default:  return String.class;
        }
    }
    
    private class TableModelRepresentation extends AbstractTableModel {
        public Object getValueAt(int row, int column) {
            return results[row][column];
        }
        
        public String getColumnName(int column) {
            return heads[column];
        }
        
        public int getRowCount() {
            return results.length;
        }
        
        public int getColumnCount() {
            return heads.length;
        }
        
        public Class getColumnClass(int column) {
            return(colClasses[column]);
        }
    }
}
