/* 
 * This is JavaSQL, a tool to check the validity of connections and queries
 * in JDBC.
 * Copyright (C) 2002 Alexander Lindhorst
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact:
 * Alexander Lindhorst
 * Elsa-Brandstrm-Weg 3
 * 33102 Paderborn
 * Germany
 * al@alexander-lindhorst.de
 */


package lindhorst.apps.jdbc.swing.modules.datamodels;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import java.sql.*;
import java.text.*;
import java.util.*;
import lindhorst.apps.jdbc.swing.*;
import lindhorst.apps.jdbc.swing.helpers.*;

public class TableDataModel implements TableModel {
    private Object[][] data=null;
    private int lastSortedIndex=-1;
    private String[] headers=null;
    private Class[] classes=null;
    private String tableName=null;
    private boolean executing=false;
    private boolean assessingSize=false;
    private Connection readConnection=null;
    private Connection writeConnection=null;
    private String warnings=null;
    private EventListenerList listeners=new EventListenerList();
    private int numberOfRecords=0;
    private int numberOfRecordsRead=0;
    
    public TableDataModel(Connection readableConnection,Connection writableConnection, String tableName) {
        if(tableName==null) throw new NullPointerException();
        if(tableName.equals("")) throw new IllegalArgumentException();
        
        this.tableName=tableName;
        readConnection=readableConnection;
        writeConnection=writableConnection;
        
        init();
    }
    
    public void init() {
        new DataRetriever().start();
    }
    
    public boolean isRetrievingData() {
        return executing;
    }
    
    public boolean isAssessingTableSize() {
        return assessingSize;
    }
    
    /*Event generation*/
    public void addChangeListener(ChangeListener listener) {
        listeners.add(ChangeListener.class,listener);
        listener.stateChanged(new ChangeEvent(this));
    }
    
    public void removeChangeListener(ChangeListener listener) {
        listeners.remove(ChangeListener.class,listener);
    }
    
    private void fireStateChanged() {
        final ChangeEvent event=new ChangeEvent(this);
        final EventListener[] array=listeners.getListeners(ChangeListener.class);
        if(SwingUtilities.isEventDispatchThread()) {
            for(int i=0;i<array.length;i++) {
                ((ChangeListener)array[i]).stateChanged(event);
            }
        }
        else { //multithreading may make this necessary
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    for(int i=0;i<array.length;i++) {
                        ((ChangeListener)array[i]).stateChanged(event);
                    }
                }
            });
        }
    }
    
    public void addTableModelListener(TableModelListener listener) {
        listeners.add(TableModelListener.class,listener);
        listener.tableChanged(new TableModelEvent(this));
    }
    
    public void removeTableModelListener(TableModelListener listener) {
        listeners.remove(TableModelListener.class,listener);
    }
    
    private void fireTableChanged(TableModelEvent event) {
        final EventListener[] array=listeners.getListeners(TableModelListener.class);
        if(SwingUtilities.isEventDispatchThread()) {
            for(int i=0;i<array.length;i++) {
                ((TableModelListener)array[i]).tableChanged(event);
            }
        }
        else { //multithreading may make this necessary
            final TableModelEvent tempEvent=event;
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    for(int i=0;i<array.length;i++) {
                        ((TableModelListener)array[i]).tableChanged(tempEvent);
                    }
                }
            });
        }
    }
    
    public void fireEntireTableChanged() {
        fireTableChanged(new TableModelEvent(this,TableModelEvent.HEADER_ROW,Integer.MAX_VALUE,TableModelEvent.ALL_COLUMNS,TableModelEvent.UPDATE));
    }
    
    private void fireTableRowsChanged(int firstRow, int lastRow) {
        fireTableChanged(new TableModelEvent(this,firstRow,lastRow));
    }
    
    private void fireTableEndChanged(int firstRow) {
        fireTableChanged(new TableModelEvent(this,firstRow));
    }
    
    private void fireTableColumnChangedInRows(int column,int firstRow, int lastRow) {
        fireTableChanged(new TableModelEvent(this,firstRow,lastRow,column,TableModelEvent.UPDATE));
    }
    /*End of event generation*/
    
    public String getTableName() {
        return tableName;
    }
    
    public String getWarnings() {
        return warnings;
    }
    
    public synchronized void setValueAt(Object value,int row,int column) {
        if(classes!=null && classes.length>column) {
            System.out.println("value: "+value.getClass()+"; target:"+classes[column].getName());
            if(value instanceof String && classes[column].getSuperclass()==Number.class)
                value=Helpers.convertToNumber((String)value,classes[column]);
            if(value instanceof String && classes[column]==java.util.Date.class)
                value=Helpers.convertToDate((String)value,Locale.getDefault());
            if(value==null) {
                fireTableColumnChangedInRows(column,row,row);
                return;
            }
        }
        Statement stmt=null;
        Connection connection=writeConnection;
        try {
            executing=true;
            connection.setReadOnly(false);
            connection.setAutoCommit(false);
            StringBuffer sqlString=new StringBuffer(1000);
            sqlString.append("UPDATE ");
            sqlString.append(tableName);
            sqlString.append(" SET ");
            sqlString.append(headers[column]);
            sqlString.append("=");
            boolean needsQuotes=getQuotationMarksNeeded(row,column);
            if(needsQuotes)sqlString.append("'");
            if(value instanceof java.util.Date)
                sqlString.append(getSQLDate((java.util.Date)value));
            else sqlString.append(value);
            if(needsQuotes)sqlString.append("'");
            boolean firstOption=true;
            int step=data[row].length/10;
            if(step<=0) step=1;
            for(int i=0;i<getColumnCount();i+=step) {
                if(data[row][i]==null) continue;
                if(data[row][i] instanceof java.util.Date) continue; //tricky!
                
                if(firstOption)sqlString.append(" WHERE ");
                if(!firstOption) sqlString.append(" AND ");
                
                firstOption=false;
                
                sqlString.append(headers[i]);
                sqlString.append("=");
                needsQuotes=getQuotationMarksNeeded(row,i);
                if(needsQuotes)sqlString.append("'");
                if(data[row][i] instanceof java.util.Date)
                    sqlString.append(getSQLDate((java.util.Date)data[row][i]));
                else sqlString.append(data[row][i]);
                if(needsQuotes)sqlString.append("'");
            }
            System.out.println(sqlString.toString());
            stmt=connection.createStatement();
            int updates=stmt.executeUpdate(sqlString.toString());
            if(updates!=1) throw new SQLException("Not exactly one record affected by change!");
            connection.commit();
            stmt.close();
            connection.setAutoCommit(true);
            
            data[row][column]=value;
        }
        catch(SQLException exception) {
            try {
                connection.rollback();
            }
            catch(Exception x) {
            }
            
            StringBuffer errors=new StringBuffer(1000);
            do {
                errors.append(exception.getMessage());
                errors.append('\n');
                
                exception=exception.getNextException();
            } while(exception!=null);
            
            Helpers.showError(new SQLException(errors.toString()));
        }
        catch(Exception e) {
            Helpers.showError(e);
        }
        finally {
            try {
                if(stmt!=null) stmt.close();
                connection.setAutoCommit(true);
            }
            catch(Exception ex) {
            }
            executing=false;
        }
        
        try {
            executing=true;
            fireTableColumnChangedInRows(column,row,row);
        }
        catch(Throwable t) {
            t.printStackTrace(System.err);
        }
        finally {
            executing=false;
        }
    }
    
    private boolean getQuotationMarksNeeded(int row, int column) {
        if(data[row][column]==null) return getQuotationMarksNeeded(classes[column]);
        else return getQuotationMarksNeeded(data[row][column]);
    }
    
    private boolean getQuotationMarksNeeded(Object object) {
        if(object==null) return false;
        else return getQuotationMarksNeeded(object.getClass());
    }
    
    private boolean getQuotationMarksNeeded(Class whatClass) {
        //System.out.println("Determining need of quotes for: "+whatClass.getName());
        if(whatClass==null) return false;
        //if(whatClass==java.util.Date.class || whatClass==java.sql.Date.class || whatClass==java.sql.Timestamp.class) return false;
        if(whatClass==Boolean.class || whatClass.getSuperclass()==Number.class)
            return false;
        else return true;
    }
    
    public Object getValueAt(int row, int column) {
        return data[row][column];
    }
    
    public boolean isCellEditable(int row, int column) {
        return !executing;
    }
    
    public int getRowCount() {
        if(data==null)return 0;
        
        return data.length;
    }
    
    public int getColumnCount() {
        if(headers==null) return 0;
        
        return headers.length;
    }
    
    public String getColumnName(int column) {
        if(headers==null || headers.length<=column) return null;
        return headers[column];
    }
    
    public String[] getColumnNames() {
        return headers;
    }
    
    public Class getColumnClass(int column) {
        return classes[column];
    }
    
    public Class[] getColumnClasses() {
        return classes;
    }
    
    /*SQLExcecution*/
    private synchronized void retrieveTableData() {
        warnings=null;
        executing=true;
        fireStateChanged(); //make cursor icon switch
        Statement stmt=null;
        ResultSet rs=null;
        Connection connection=readConnection;
        
        synchronized(connection) {
            try {
                assessingSize=true;
                fireStateChanged();
                connection.setAutoCommit(true);
                stmt=connection.createStatement();
                rs=stmt.executeQuery("select count(*) from "+tableName);
                //no checks, always holds a  value
                if(rs.next())
                    numberOfRecords=rs.getInt(1);
                rs.close();
                
                //For GUI feedback we want max 10 GUI updates while
                //retrieving data
                int interval=numberOfRecords/10;
                if(interval<1) {
                    //happens with small tables with less than 10 records
                    interval=1;
                }
                
                assessingSize=false;
                fireStateChanged();
                
                rs=stmt.executeQuery("select * from "+tableName);
                ResultSetMetaData meta=rs.getMetaData();
                int size=meta.getColumnCount();
                
                //get headers
                headers=new String[size];
                for(int i=1;i<=size;i++)
                    headers[i-1]=meta.getColumnName(i);
                
                //get Classes
                classes=new Class[size];
                for(int i=1;i<=size;i++)
                    classes[i-1]=getClassForColumn(meta.getColumnType(i));
                
                ArrayList array=new ArrayList(1000);
                Object[] record=null;
                while(rs.next()) {
                    record=new Object[size];
                    for(int i=1;i<=size;i++)
                        record[i-1]=rs.getObject(i);
                    
                    array.add(record);
                    numberOfRecordsRead++;
                    if(numberOfRecordsRead%interval==0) {
                        fireStateChanged();
                        fireTableRowsChanged(numberOfRecordsRead-interval,numberOfRecordsRead);
                    }
                }
                
                Object[] temp=array.toArray();
                data=new Object[temp.length][];
                System.arraycopy(temp,0,data,0,data.length);
                rs.close();
                stmt.close();
                rs=null;
                stmt=null;
            }
            catch(Exception sqle) {
                Helpers.showError(sqle);
                
                if(sqle instanceof SQLException) {
                    StringBuffer buffer=new StringBuffer(1000);
                    do {
                        buffer.append(sqle.getMessage());
                        sqle=((SQLException)sqle).getNextException();
                    } while(sqle!=null);
                    
                    warnings=buffer.toString();
                }
            }
            finally {
                try {
                    if(rs!=null) rs.close();
                    if(stmt!=null) stmt.close();
                }
                catch(Exception e) {
                }
                fireEntireTableChanged();
                fireStateChanged();
                executing=false;
            }
        }
    }
    
    public synchronized void deleteLines(int[] lineIndices) {
        if(lineIndices==null || lineIndices.length==0) return;
        
        //number of conditions
        int step=headers.length/10; //10 conditions
        if(step<=0) step=1;
        
        //keep track of smallest index
        int smallest=0;
        
        //keep Track of errors
        StringBuffer exceptions=new StringBuffer(1000);
        //build sql command
        StringBuffer sql=null;
        //can we do it
        boolean conditionSet=false;
        
        Connection connection=writeConnection;
        Statement stmt=null;
        try {
            connection.setReadOnly(false);
            connection.setAutoCommit(true);  //deletes ok in auto commit mode
            stmt=connection.createStatement();
        }
        catch(Exception e) {
            Helpers.showError(e);
            fireEntireTableChanged();
            return;
        }
        
        for(int i=0;i<lineIndices.length;i++) {
            if(lineIndices[i]<smallest)smallest=lineIndices[i];
            //build command
            sql=new StringBuffer(300);
            sql.append("DELETE ");
            //sql.append(tableName);
            sql.append(" FROM ");
            sql.append(tableName);
            
            conditionSet=false;
            
            for(int j=0;j<data[lineIndices[i]].length;j+=step) {
                if(data[lineIndices[i]][j]==null) continue;
                if(data[lineIndices[i]][j] instanceof java.util.Date) continue;
                
                if(conditionSet) sql.append(" AND");
                
                if(!conditionSet) sql.append(" WHERE");
                conditionSet=true;
                
                sql.append(" ");
                sql.append(headers[j]);
                sql.append("=");
                if(getQuotationMarksNeeded(lineIndices[i],j)) sql.append("'");
                if(data[lineIndices[i]][j] instanceof java.util.Date)
                    sql.append(getSQLDate((java.util.Date)data[lineIndices[i]][j]));
                else sql.append(data[lineIndices[i]][j]);
                if(getQuotationMarksNeeded(lineIndices[i],j)) sql.append("'");
            }
            
            System.out.println(sql);
            
            //done; shove it into the database
            synchronized(connection) {
                try {
                    //executeSingleRecordCommand(sql.toString());
                    boolean success=stmt.execute(sql.toString());
                    if(success) throw new SQLException("Unexpectly received a record set from the database!");
                    int updated=stmt.getUpdateCount();
                    if(updated>1) throw new SQLException("More than one record affected by a command intended to affect just one record!");
                    connection.commit();
                    data[lineIndices[i]]=null;
                }
                catch(SQLException sqle) {
                    try {
                        connection.rollback();
                    }
                    catch(Exception e) { }
                    do {
                        exceptions.append(sqle.getMessage());
                        exceptions.append('\n');
                        sqle=sqle.getNextException();
                    } while (sqle!=null);
                }
            }
        }
        
        try {
            stmt.close();
        }
        catch(Exception e){ }
        
        //data komprimieren
        int oldLength=data.length;
        Object[][] temp=new Object[data.length][];
        int nulls=0;
        int tempIndex=0;
        synchronized(data) {
            for(int i=0;i<data.length;i++) {
                if(data[i]!=null) {
                    temp[tempIndex]=data[i];
                    tempIndex++;
                }
                else {
                    nulls++;
                }
            }
            //copy back
            data=new Object[data.length-nulls][];
            System.arraycopy(temp,0,data,0,data.length);
        }
        
        String sqlErrors=exceptions.toString();
        if(sqlErrors.length()>0) {
            JOptionPane.showMessageDialog(Helpers.getTopLevelContainer(),sqlErrors,"Fehler",JOptionPane.WARNING_MESSAGE);
        }
        
        fireEntireTableChanged();
    }
    
    public void insertLine(Object[] lineData) throws SQLException {
        if(lineData==null) return;
        if(headers==null) return;
        if(lineData.length!=headers.length) throw new IllegalArgumentException("Data does not fit into model!");
        
        StringBuffer buffer=new StringBuffer(1000);
        buffer.append("INSERT INTO ");
        buffer.append(tableName);
        buffer.append(" (");
        boolean first=true;
        boolean needMarks=false;
        for(int i=0;i<headers.length;i++) {
            if(lineData[i]==null) continue;
            if(!first)buffer.append(",");
            else
                first=false;
            buffer.append(headers[i]);
        }
        buffer.append(") VALUES (");
        first=true;
        for(int i=0;i<lineData.length;i++) {
            if(lineData[i]==null) continue;
            if(!first)buffer.append(",");
            else
                first=false;
            needMarks=getQuotationMarksNeeded(lineData[i]);
            if(needMarks) buffer.append("'");
            if(lineData[i] instanceof java.util.Date)
                buffer.append(getSQLDate((java.util.Date)lineData[i]));
            else buffer.append(lineData[i]);
            if(needMarks) buffer.append("'");
        }
        buffer.append(")");
        
        System.out.println(buffer);
        
        SQLException e=null;
        
        Connection connection=writeConnection;
        connection.setAutoCommit(false);
        Statement stmt=null;
        
        try {
            stmt=connection.createStatement();
            int updated=stmt.executeUpdate(buffer.toString());
            if(updated!=1)
                throw new SQLException("Not exactly one record affected by change!");
            connection.commit();
        }
        catch(SQLException sqle) {
            e=sqle;
            connection.rollback();
        }
        finally {
            stmt.close();
            connection.setAutoCommit(true);
        }
        
        if(e!=null) throw e;
        
        //database accepted it, put it into the array
        Object[][] temp=data;
        data=new Object[data.length+1][];
        System.arraycopy(temp,0,data,0,temp.length);
        int insertIndex=data.length-1;
        data[insertIndex]=lineData;
        
        fireTableChanged(new TableModelEvent(this,insertIndex,insertIndex,TableModelEvent.ALL_COLUMNS,TableModelEvent.INSERT));
    }
    
    public int getNumberOfRecords() {
        return numberOfRecords;
    }
    
    public int getNumberOfRecordsRead() {
        return numberOfRecordsRead;
    }
    
    public void doSort(String columnName) {
        if(!executing) {
            new TableSorter(columnName);
        }
    }
    
    private static Class getClassForColumn(int type) {
        switch(type) {
            case Types.BIGINT: return Number.class;
            case Types.BIT: return Boolean.class;
            case Types.CHAR:  return String.class;
            case Types.DATE:  return java.util.Date.class;
            case Types.DECIMAL: return Double.class;
            case Types.DOUBLE:  return Double.class;
            case Types.FLOAT: return Float.class;
            case Types.INTEGER: return Integer.class;
            case Types.LONGVARCHAR: return String.class;
            case Types.NUMERIC: return Number.class;
            case Types.REAL:  return Number.class;
            case Types.SMALLINT: return Integer.class;
            case Types.TIME:  return java.util.Date.class;
            case Types.TIMESTAMP: return java.util.Date.class;
            case Types.TINYINT: return Integer.class;
            
            default:  return String.class;
        }
    }
    
    private static String getSQLDate(java.util.Date date) {
        //System.out.println("Getting SQLDate for "+date.getClass().getName());
        StringBuffer buffer=new StringBuffer(30); //gimme the long type
        //buffer.append('#');
        DateFormat formatter=DateFormat.getDateInstance(DateFormat.SHORT,Locale.US);
        buffer.append(formatter.format(date));
        return buffer.toString();
    }
    
    private class DataRetriever extends Thread {
        DataRetriever() {
            setName("Data Retriever for table "+tableName);
            setPriority((Thread.MIN_PRIORITY+Thread.NORM_PRIORITY)/2);
            setDaemon(true);
        }
        
        public void run() {
            retrieveTableData();
        }
    }
    
    private class TableSorter extends Thread {
        private String headerValue=null;
        private int columnIndex=-1;
        
        TableSorter(String headerValue) {
            if(headerValue==null) {
                System.out.println("headerValue==null");
                return;
            }
            
            this.headerValue=headerValue;
            setPriority(Thread.MAX_PRIORITY);
            setDaemon(false);
            start();
        }
        
        public void run() {
            executing=true;
            fireStateChanged();
            
            columnIndex=getColumnIndex();
            if(columnIndex<0) {
                System.out.println("columnIndex<0");
                executing=false;
                fireStateChanged();
                return;
            }
            
            Object[] values=new Object[data.length];
            for(int i=0;i<values.length;i++)
                values[i]=data[i][columnIndex];
            
            if(values==null||values.length==0) {
                System.out.println("values==null||values.length==0");
                executing=false;
                fireStateChanged();
                return;
            }
            
            int[] indices=null;
            boolean reverse=lastSortedIndex==(columnIndex+1);
            
            if(values[0] instanceof Number) {
                Number[] numbers=new Number[values.length];
                System.arraycopy(values,0,numbers,0,values.length);
                indices=ArraySorter.sortNumbers(numbers);
            }
            else if(values[0] instanceof java.util.Date) {
                java.util.Date[] dates=new java.util.Date[values.length];
                System.arraycopy(values,0,dates,0,values.length);
                indices=ArraySorter.sortDates(dates);
            }
            else if(values[0] instanceof java.lang.String) {
                String[] strings=new String[values.length];
                System.arraycopy(values,0,strings,0,values.length);
                indices=ArraySorter.sortStrings(strings);
            }
            else {
                indices=ArraySorter.sortObjects(values);
            }
            
            if(indices!=null) {
                if(reverse)indices=ArraySorter.invert(indices);
                establishNewOrder(indices);
            }
            
            lastSortedIndex=0;
            if(reverse)
                lastSortedIndex-=(columnIndex+1);
            else
                lastSortedIndex+=(columnIndex+1);
            
            executing=false;
            fireStateChanged();
        }
        
        private int getColumnIndex() {
            int index=-1;
            for(int i=0;i<headers.length;i++) {
                if(headers[i].equals(headerValue)) {
                    index=i;
                    break;
                }
            }
            return index;
        }
        
        private void establishNewOrder(int[] indices) {
            synchronized(data) {
                Object[][] values=new Object[data.length][];
                for(int i=0;i<data.length;i++)
                    values[i]=data[indices[i]];
                data=values;
            }
            fireTableRowsChanged(0,data.length-1);
        }
    }
}

