/* 
 * This is JavaSQL, a tool to check the validity of connections and queries
 * in JDBC.
 * Copyright (C) 2002 Alexander Lindhorst
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact:
 * Alexander Lindhorst
 * Elsa-Brandstrm-Weg 3
 * 33102 Paderborn
 * Germany
 * al@alexander-lindhorst.de
 */
/*
 * UniversalDriverWrapper.java
 *
 * Created on 18. November 2002, 16:22
 */

package lindhorst.apps.jdbc.swing.sql;

import java.sql.*;
import java.util.*;

/**
 *
 * @author  lindhrst
 */
public class UniversalDriverWrapper implements java.sql.Driver {
    
    private static Set registry=new HashSet();
    
    private Driver wrappedDriver;
    
    public UniversalDriverWrapper(ClassLoader loader,String wrappedDriverClass)
    throws Exception {
        wrappedDriver=
        (Driver) Class.forName(wrappedDriverClass,true,loader).newInstance();
        DriverManager.registerDriver(this);
        registry.add(wrappedDriverClass);
    }
    
    /** Retrieves whether the driver thinks that it can open a connection
     * to the given URL.  Typically drivers will return <code>true</code> if they
     * understand the subprotocol specified in the URL and <code>false</code> if
     * they do not.
     *
     * @param url the URL of the database
     * @return <code>true</code> if this driver understands the given URL;
     *         <code>false</code> otherwise
     * @exception SQLException if a database access error occurs
     */
    public boolean acceptsURL(String url) throws SQLException {
        return wrappedDriver.acceptsURL(url);
    }
    
    /** Attempts to make a database connection to the given URL.
     * The driver should return "null" if it realizes it is the wrong kind
     * of driver to connect to the given URL.  This will be common, as when
     * the JDBC driver manager is asked to connect to a given URL it passes
     * the URL to each loaded driver in turn.
     *
     * <P>The driver should throw an <code>SQLException</code> if it is the right
     * driver to connect to the given URL but has trouble connecting to
     * the database.
     *
     * <P>The <code>java.util.Properties</code> argument can be used to pass
     * arbitrary string tag/value pairs as connection arguments.
     * Normally at least "user" and "password" properties should be
     * included in the <code>Properties</code> object.
     *
     * @param url the URL of the database to which to connect
     * @param info a list of arbitrary string tag/value pairs as
     * connection arguments. Normally at least a "user" and
     * "password" property should be included.
     * @return a <code>Connection</code> object that represents a
     *         connection to the URL
     * @exception SQLException if a database access error occurs
     */
    public Connection connect(String url, java.util.Properties info) throws SQLException {
        return wrappedDriver.connect(url,info);
    }
    
    /** Retrieves the driver's major version number. Initially this should be 1.
     *
     * @return this driver's major version number
     */
    public int getMajorVersion() {
        return wrappedDriver.getMajorVersion();
    }
    
    /** Gets the driver's minor version number. Initially this should be 0.
     * @return this driver's minor version number
     */
    public int getMinorVersion() {
        return wrappedDriver.getMinorVersion();
    }
    
    /** Gets information about the possible properties for this driver.
     * <P>
     * The <code>getPropertyInfo</code> method is intended to allow a generic
     * GUI tool to discover what properties it should prompt
     * a human for in order to get
     * enough information to connect to a database.  Note that depending on
     * the values the human has supplied so far, additional values may become
     * necessary, so it may be necessary to iterate though several calls
     * to the <code>getPropertyInfo</code> method.
     *
     * @param url the URL of the database to which to connect
     * @param info a proposed list of tag/value pairs that will be sent on
     *          connect open
     * @return an array of <code>DriverPropertyInfo</code> objects describing
     *          possible properties.  This array may be an empty array if
     *          no properties are required.
     * @exception SQLException if a database access error occurs
     */
    public DriverPropertyInfo[] getPropertyInfo(String url, java.util.Properties info) throws SQLException {
        return wrappedDriver.getPropertyInfo(url,info);
    }
    
    /** Reports whether this driver is a genuine JDBC
     * Compliant<sup><font size=-2>TM</font></sup> driver.
     * A driver may only report <code>true</code> here if it passes the JDBC
     * compliance tests; otherwise it is required to return <code>false</code>.
     * <P>
     * JDBC compliance requires full support for the JDBC API and full support
     * for SQL 92 Entry Level.  It is expected that JDBC compliant drivers will
     * be available for all the major commercial databases.
     * <P>
     * This method is not intended to encourage the development of non-JDBC
     * compliant drivers, but is a recognition of the fact that some vendors
     * are interested in using the JDBC API and framework for lightweight
     * databases that do not support full database functionality, or for
     * special databases such as document information retrieval where a SQL
     * implementation may not be feasible.
     * @return <code>true</code> if this driver is JDBC Compliant; <code>false</code>
     *         otherwise
     */
    public boolean jdbcCompliant() {
        return wrappedDriver.jdbcCompliant();
    }
    
    /**
     *Retrieves all drivers registered with the DriverManager as well as all
     *driver having been loaded through instances of this class as an array
     *of class names. Note: Instances of this class are left
     *out since it only serves as a wrapper.
     *@return array of class names
     */
    public static Object[] getRegisteredDrivers() {
        Enumeration e=DriverManager.getDrivers();
        String driverClass=null;
        while(e.hasMoreElements()){
            driverClass=e.nextElement().getClass().getName();
            if(!driverClass.startsWith("lindhorst.apps.jdbc.swing.sql")) {
                registry.add(driverClass);
            }
        }
        return registry.toArray();
    }
    
    /**
     *Adds the given class name to the registry of known driver classes.
     *<b>Note:</b> No checks regarding the existence of the given class are
     *performed.
     *@return className the class name to be registered
     **/
    public static void registerDriverClassName(String className) {
        if(className==null) {
            return;
        }
        registry.add(className);
    }
    
}
